<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\ACommonController;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class PaymentManagementController extends ACommonController
{

    public function index(Request $request)
    {

        if ($request->isMethod('post')) {

            $action = $request->input('action_type');

            if ($action == 'search') {
                session([
                    'dsearch.from_date'   => $request->input('from_date'),
                    'dsearch.to_date'     => $request->input('to_date'),
                    'dsearch.driver_name' => $request->input('driver_name'),
                    'dsearch.status1'     => $request->input('status1'),
                ]);
            } elseif ($action == 'clear') {
                Session::forget('dsearch');
            } elseif ($action == 'reject') {
                // handle reject
            } elseif ($action == 'delete') {
                // handle delete
            }

        }

        $sfrom_date   = session('dsearch.from_date', date('d-m-Y'));
        $sto_date     = session('dsearch.to_date', date('d-m-Y'));
        $sdriver_name = session('dsearch.driver_name', '');
        $sstatus1     = session('dsearch.status1', 'all');

        $query = DB::table('trips')
            ->join('driver_management', 'trips.driver_id', '=', 'driver_management.id')
            ->select(
                'trips.*',
                'driver_management.driver_code',
                'driver_management.driver_name'
            );

        // Date filter
        if ($sfrom_date) {
            $sfrom_date_db = Carbon::createFromFormat('d-m-Y', $sfrom_date)->format('Y-m-d');
            $query->whereDate('trips.trip_date', '>=', $sfrom_date_db);
        }

        if ($sto_date) {
            $sto_date_db = Carbon::createFromFormat('d-m-Y', $sto_date)->format('Y-m-d');
            $query->whereDate('trips.trip_date', '<=', $sto_date_db);
        }

        if ($sdriver_name && $sdriver_name > 0) {
            $search = $sdriver_name;
            $query->where(function ($q) use ($search) {

                $q->where('driver_management.id', $search);
                // $q->where('driver_management.driver_name', 'like', "%{$search}%")
                //     ->orWhere('driver_management.driver_code', 'like', "%{$search}%");
            });
        }

        $driverList = DB::table('driver_management')
            ->orderBy('driver_name', 'asc')
            ->orderBy('driver_code', 'asc')
            ->select('id', 'driver_code', 'driver_name', 'user_name')
            ->get();

        // Payment status filter (0 = Unpaid, 1 = Pending, 2 = Paid)
        if ($sstatus1 && $sstatus1 !== 'all') {
            switch ($sstatus1) {
                case 'unpaid':
                    $query->where('trips.payment_status', 0);
                    break;
                case 'partially_paid':
                    $query->where('trips.payment_status', 1);
                    break;
                case 'paid':
                    $query->where('trips.payment_status', 2);
                    break;
            }
        }

        $datas = $query->orderByDesc('trips.created_at')->get();

        // Generate PDF if requested
        if ($request->has('download') && $request->download === 'pdf') {
            $pdf = Pdf::loadView('admin.payment_management.driver_report_pdf', [
                'datas'        => $datas,
                'from_date'    => $request->from_date,
                'to_date'      => $request->to_date,
                'sfrom_date'   => $sfrom_date,
                'sto_date'     => $sto_date,
                'sdriver_name' => $sdriver_name,
                'sstatus1'     => $sstatus1,
                'driverList'   => $driverList,
            ]);
            // return $pdf->stream('driver-report.pdf');

            return $pdf->download('driver_report.pdf');
        }

        return view('admin.payment_management.list', [
            'datas'        => $datas,
            'from_date'    => $request->from_date,
            'to_date'      => $request->to_date,
            'sfrom_date'   => $sfrom_date,
            'sto_date'     => $sto_date,
            'sdriver_name' => $sdriver_name,
            'sstatus1'     => $sstatus1,
            'driverList'   => $driverList,
        ]);

    }

    public function create()
    {
        $drivers = DB::table('driver_management')->get(); // For driver dropdown
        return view('admin.payment_management.add', compact('drivers'));
    }
    // public function store(Request $request)
    // {
    //     $validated = $request->validate([
    //         'driver_id' => 'required|exists:driver_management,id',
    //         'trip_date' => 'required|date',
    //         'amount'    => 'required|numeric',
    //         'receipt'   => 'required|file|mimes:jpg,jpeg,png,pdf',
    //         'status'    => 'required|in:0,1,2',
    //     ]);

    //     // Store receipt file
    //     $receiptPath = $request->file('receipt')->store('payment_receipts', 'public');

    //     // Insert into your payment table (e.g., driver_payments)
    //     DB::table('driver_payments')->insert([
    //         'driver_id'      => $validated['driver_id'],
    //         'trip_date'      => $validated['trip_date'],
    //         'amount'         => $validated['amount'],
    //         'receipt'        => $receiptPath,
    //         'payment_status' => $validated['status'],
    //         'created_at'     => now(),
    //         'updated_at'     => now(),
    //     ]);

    //     return redirect()->route('admin.PaymentManagement-index')->with('success', 'Payment added successfully!');
    // }

    public function store(Request $request)
    {
        $request->validate([
            'driver_id'         => 'required|exists:driver_management,id',
            'trip_date'         => 'required|date',
            'amount'            => 'required|numeric',
            'commission_amount' => 'required|numeric',
            'paid_amount'       => 'required|numeric',
            'payment_receipt'   => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        // 1. Find the trip ID using driver and date
        $trip = DB::table('trips')
            ->where('driver_id', $request->driver_id)
            ->whereDate('trip_date', $request->trip_date)
            ->first();

        if (! $trip) {
            return redirect()->back()->with('error', 'No trip found for this driver on the selected date.');
        }

        // 2. Handle file upload
        $receiptPath = null;
        if ($request->hasFile('receipt')) {
            $receiptPath = $request->file('receipt')->store('receipts', 'public');
        }

        // 3. Insert into trip_payment_details
        DB::table('trip_payment_details')->insert([
            'trip_id'                   => $trip->id,
            'req_amount'                => $request->amount,
            'payment_receipt'           => $receiptPath,
            'paid_amount'               => $request->paid_amount,
            'pay_status'                => 2, // ✅ Add this line to match calculatetrippayment logic
            'payment_status_approvedBy' => auth()->user()->id ?? null,
            'approved_time'             => Carbon::now(),
            'created_at'                => Carbon::now(),
            'updated_at'                => Carbon::now(),
        ]);

        $this->calculatetrippayment($trip->id);
        return redirect()->route('admin.PaymentManagement-index')->with('success', 'Payment saved successfully.');
    }
    public function approvalwaiting(Request $request)
    {
        $status = $request->input('status1', 'all');

        $dataQuery1 = DB::table('trip_payment_details')
            ->join('trips', 'trip_payment_details.trip_id', '=', 'trips.id')
            ->join('driver_management', 'trips.driver_id', '=', 'driver_management.id');

        if ($status == 'pending') {
            $dataQuery1->where('trip_payment_details.pay_status', 0);
        } elseif ($status == 'approved') {
            $dataQuery1->where('trip_payment_details.pay_status', 2);
        } elseif ($status == 'rejected') {
            $dataQuery1->where('trip_payment_details.pay_status', 3);

        }

        $datas = $dataQuery1->select(
            'trip_payment_details.id',
            'driver_management.driver_name',
            'driver_management.driver_code',
            'trip_payment_details.req_amount',
            'trip_payment_details.paid_amount',
            'trip_payment_details.payment_receipt',
            'trips.pending_amount',
            'trip_payment_details.pay_status'
        )
            ->orderBy('trip_payment_details.pay_status', 'asc')
            ->orderBy('trip_payment_details.created_at', 'asc')
            ->get();

        return view('admin.payment_management.approval_waiting', compact('datas'));
    }

    public function updatePayStatus(Request $request)
    {
        $id         = $request->input('id');
        $status     = $request->input('status'); // 2 for approved, 3 for rejected
        $tripDetail = DB::table('trip_payment_details')->where('id', $id)->where('pay_status', 2)->first();
        if (! $tripDetail) {
            return response()->json([
                'success' => false,
                'message' => 'No record found',
            ]);
        }
        $updated = DB::table('trip_payment_details')
            ->where('id', $id)
            ->where('pay_status', 2)
            ->update(['pay_status' => $status]);
        if ($updated) {

            $this->calculatetrippayment($tripDetail->trip_id);
            return response()->json([
                'success' => true,
                'message' => 'Status updated successfully',
            ]);

        } else {
            return response()->json([
                'success' => false,
                'message' => 'No rows affected',
            ]);
        }

    }
    public function calculatetrippayment($tripId)
    {
        // 1. Get sum of payments for this trip
        $sumAmount = DB::table('trip_payment_details')
            ->where('trip_id', $tripId)
            ->where('pay_status', 2)
            ->sum('paid_amount');

        // 2. Get commission amount for this trip
        $trip = DB::table('trips')->where('id', $tripId)->first();

        if (! $trip) {
            // return response()->json(['success' => false, 'message' => 'Trip not found']);
            return 0;
        }
        $pending_amount = $trip->pending_amount;
        $commission     = $trip->commission_amount;
        $paymentStatus  = 'unpaid';

        // 3. Determine payment status
        if ($sumAmount >= $commission) {
            $pending_amount = 0;
            $paymentStatus  = '2';
        } elseif ($sumAmount == 0) {
            $pending_amount = $commission;
            $paymentStatus  = '0';
        } elseif ($sumAmount < $commission) {
            $pending_amount = $commission - $sumAmount;
            $paymentStatus  = '1';
        }

        // 4. Update trip payment status
        DB::table('trips')->where('id', $tripId)->update([
            'payment_status' => $paymentStatus,
            'paid_amount'    => $sumAmount,
            'paid_at'        => now(),
            'pending_amount' => $pending_amount,

        ]);

        return response()->json([
            'success'        => true,
            'message'        => 'Payment status updated successfully',
            'total_paid'     => $sumAmount,
            'payment_status' => $paymentStatus,
        ]);
    }
    public function getTripData(Request $request)
    {
        $driver_id = $request->input('driver_id');
        $trip_date = $request->input('trip_date');

        $trip = DB::table('trips')
            ->where('driver_id', $driver_id)
            ->whereDate('trip_date', $trip_date)
            ->first();

        if ($trip) {
            return response()->json([
                'success'            => true,
                'amount'             => $trip->amount ?? 0,
                'commission_amount'  => $trip->commission_amount ?? 0,
                'driver_paid_amount' => $trip->driver_paid_amount ?? 0,
                'balance_amount'     => $trip->balance_amount ?? 0,
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'No trip data found for this driver and date.',
            ]);
        }
    }

}
