<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class StartRideController extends Controller
{

    public function StartRide(Request $request)
    {
        $timeNow = date('Y-m-d H:i:s');
        $request->validate([
            'booking_id'      => 'required|exists:otp_codes,booking_id',
            'start_image'     => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'start_km'        => 'required|numeric',
            'start_latitude'  => 'required|numeric',
            'start_longitude' => 'required|numeric',
        ]);

        $driver = auth('api_driver')->user();
        try {

            $bookingRow = DB::table('booking_management')->where('id', $request->booking_id)->where('deleted_status', 0)->where('driver_id', $driver->id)->first();

            $data = [
                'start_km'        => $request->start_km,
                'start_latitude'  => $request->start_latitude,
                'start_longitude' => $request->start_longitude,
                'start_date'      => $timeNow,
                'updated_at'      => now(),
            ];
            $imagePath = '';
            // Handle file upload if present
            if ($request->hasFile('start_image')) {
                $imagePath           = $request->file('start_image')->store('start_images', 'public');
                $data['start_image'] = $imagePath;
            }

            DB::table('otp_codes')
                ->where('booking_id', $request->booking_id)
                ->where('driver_id', $driver->id)
                ->update($data);

            DB::table('booking_management')
                ->where('id', $request->booking_id)
                ->where('driver_id', $driver->id)
                ->where('deleted_status', 0)
                ->update([
                    'trip_status'    => 1,
                    'booking_status' => 1,
                    'updated_at'     => now(),
                ]);

            DB::commit();

            $driver_arrivedRow = DB::table('driver_arrived')->where('id', $bookingRow->arrived_id)->first();

            if ($driver_arrivedRow) {

                // 1. total_km as start_km + end_km
                $total_km = $request->start_km - $driver_arrivedRow->start_km;

                // 2. Calculate lat-lon distance using Haversine formula
                $lat1 = $driver_arrivedRow->start_latitude;
                $lon1 = $driver_arrivedRow->start_longitude;
                $lat2 = $request->start_latitude;
                $lon2 = $request->start_longitude;

                $earthRadius = 6371; // in kilometers

                $dLat = deg2rad($lat2 - $lat1);
                $dLon = deg2rad($lon2 - $lon1);

                $a = sin($dLat / 2) * sin($dLat / 2) +
                cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
                sin($dLon / 2) * sin($dLon / 2);

                $c               = 2 * atan2(sqrt($a), sqrt(1 - $a));
                $lat_lon_totalkm = $earthRadius * $c;

                // 3. Calculate amount
                $amount_per_km       = 0;
                $lat_lon_totalamount = $lat_lon_totalkm * $amount_per_km;
                $total_amount        = $total_km * $amount_per_km;

                DB::table('driver_arrived')->where('id', $driver_arrivedRow->id)->update([
                    'driver_arrived_at'   => now(),
                    'end_latitude'        => $lat2,
                    'end_longitude'       => $lon2,
                    'end_image'           => $imagePath,
                    'end_km'              => $request->start_km,
                    'total_km'            => $total_km,
                    'amount_per_km'       => $amount_per_km,
                    'total_amount'        => $total_amount,
                    'lat_lon_totalkm'     => $lat_lon_totalkm,
                    'lat_lon_totalamount' => $lat_lon_totalamount,
                    'updated_at'          => now(),
                ]);

            }

            return response()->json(['message' => 'Ride start data updated successfully.']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to update ride start data.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    // public function EndRide(Request $request)
    // {
    //     DB::beginTransaction();

    //     try {

    //         $driver = auth('api_driver')->user();

    //         $bookingId = $request->booking_id;

    //         if ($bookingId > 0) {

    //             $booking = DB::table('booking_management')
    //                 ->select('id', 'cartype_id', 'ac_type', 'traveltype_id', 'driver_id', 'package_id')
    //                 ->where('id', $bookingId)
    //                 ->where('driver_id', $driver->id)
    //                 ->first();

    //             if (! $booking) {
    //                 return response()->json(['success' => false, 'message' => 'Booking or trip data not found'], 404);
    //             }

    //         } else {
    //             return response()->json(['success' => false, 'message' => 'Please Enter Valid Booking Id'], 404);
    //         }

    //         // Step 1: Update OTP end details
    //         $data = [
    //             'end_km'        => $request->end_km,
    //             'end_latitude'  => $request->end_latitude,
    //             'end_longitude' => $request->end_longitude,
    //             'end_date'      => now(),
    //             'updated_at'    => now(),
    //         ];

    //         if ($request->hasFile('end_image')) {
    //             $data['end_image'] = $request->file('end_image')->store('end_images', 'public');
    //         }
    //         DB::table('otp_codes')->where('booking_id', $bookingId)->update($data);

    //         // Step 2: Update trip and driver status
    //         DB::table('booking_management')->where('id', $bookingId)->update([
    //             'booking_status' => 4, // End
    //             'updated_at'     => now(),
    //         ]);

    //         DB::table('driver_management')->where('id', $booking->driver_id)->update([
    //             'trip_status' => 0, // Off trip
    //             'updated_at'  => now(),
    //         ]);
    //         $booking = DB::table('booking_management')
    //             ->select('id', 'cartype_id', 'ac_type', 'traveltype_id', 'driver_id', 'package_id')
    //             ->where('id', $bookingId)
    //             ->where('driver_id', $driver->id)
    //             ->first();

    //         DB::table('driver_management')->where('id', $booking->driver_id)->update(['trip_status' => 0, 'updated_at' => now()]);

    //         // Step 3: Fetch OTP and calculate distance
    //         $otp = DB::table('otp_codes')->where('booking_id', $bookingId)->first();
    //         if (! $otp || ! $booking) {
    //             DB::rollBack();
    //             if (! $otp || ! $booking) {
    //                 Log::error('Store Error: No otp');
    //             } else {
    //                 Log::error('Store Error: No booking');
    //             }

    //             return response()->json(['success' => false, 'message' => 'Booking or trip data not found'], 404);
    //         }

    //         $traveledKm    = $totalKm    = round($otp->end_km - $otp->start_km, 2);
    //         $cartype_id    = $booking->cartype_id;
    //         $traveltype_id = $booking->traveltype_id;

    //         $setting = DB::table('settings')->where('id', 1)->first();
    //         $package = DB::table('packages as p')
    //             ->join('outstation_package as op', function ($join) use ($cartype_id, $traveltype_id) {
    //                 $join->on('p.id', '=', 'op.package_id')
    //                     ->where('op.cartype_id', $cartype_id)
    //                     ->where('op.inout_type', $traveltype_id);
    //             })
    //             ->where('p.traveltype_id', $traveltype_id)
    //             ->where('p.km', '<=', $traveledKm)
    //             ->orderBy('p.km', 'desc')
    //             ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'p.free_hours as free_hours', 'op.ac_price', 'op.additionalperkm_ac_price', 'op.additionalperkm_nonac_price', 'op.nonac_price', 'op.local_type', 'op.per_hours_ac_price', 'op.per_hours_price')
    //             ->first();

    //         if (! $package) {

    //             $package = DB::table('packages as p')
    //                 ->join('outstation_package as op', function ($join) use ($cartype_id, $traveltype_id) {
    //                     $join->on('p.id', '=', 'op.package_id')
    //                         ->where('op.cartype_id', $cartype_id)
    //                         ->where('op.inout_type', $traveltype_id);
    //                 })
    //                 ->where('p.traveltype_id', $traveltype_id)
    //                 ->orderBy('p.km', 'asc')
    //                 ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'p.free_hours as free_hours', 'op.ac_price', 'op.additionalperkm_ac_price', 'op.additionalperkm_nonac_price', 'op.nonac_price', 'op.local_type', 'op.per_hours_ac_price', 'op.per_hours_price')
    //                 ->first();

    //         }

    //         if (! $package) {
    //             DB::rollBack();
    //             Log::error('Store Error: No Pack Found');
    //             return response()->json(['success' => false, 'message' => 'No Pack Found'], 404);
    //         }
    //         DB::table('booking_management')->where('id', $bookingId)->update(['package_id' => $package->package_id]);

    //         $tripInfo = DB::table('otp_codes as oc')
    //             ->join('booking_management as bm', 'oc.booking_id', '=', 'bm.id')
    //             ->join('driver_management as dm', 'bm.driver_id', '=', 'dm.id')
    //             ->join('vehicle_management as vm', 'dm.vehicle_id', '=', 'vm.id')
    //             ->join('cartype_management as cm', 'vm.cartype_id', '=', 'cm.id')
    //             ->join('packages as pk', 'pk.id', '=', 'bm.package_id')
    //             ->select(
    //                 'bm.customer_name',
    //                 'bm.from_address',
    //                 'bm.to_address',
    //                 'oc.start_km',
    //                 'oc.end_km',
    //                 'pk.km as included_km',
    //                 'cm.car_type',
    //                 'oc.created_at as start_time',
    //                 'oc.updated_at as end_time'
    //             )
    //             ->where('oc.booking_id', $bookingId)
    //             ->first();
    //         if (! $tripInfo) {
    //             DB::rollBack();
    //             Log::error('Store Error: Failed to end ride and save fare.');
    //             return response()->json(['success' => false, 'message' => 'aFailed to end ride and save fare.'], 404);

    //             DB::rollBack();
    //             Log::error('Store Error: No tripInfo');
    //             return response()->json(['success' => false, 'message' => 'Trip info not found'], 404);
    //         }
    //         $startTime             = Carbon::parse($tripInfo->start_time);
    //         $endTime               = Carbon::parse($tripInfo->end_time);
    //         $travelled_time_in_min = $endTime->diffInMinutes($startTime);
    //         $travelled_time_in_hr  = $travelled_time_in_min / 60;
    //         $isMinutes             = false;

    //         $rate             = $booking->ac_type === 'AC' ? $package->ac_price : $package->nonac_price;
    //         $additionalKMrate = $booking->ac_type === 'AC' ? $package->additionalperkm_ac_price : $package->additionalperkm_nonac_price;
    //         $additionalhrrate = $booking->ac_type === 'AC' ? $package->per_hours_ac_price : $package->per_hours_price;
    //         $includedKm       = $package->included_km;
    //         $includedHr       = $package->free_hours;
    //         $extraKm          = max(0, $traveledKm - $includedKm);
    //         $extrahrs         = 0;

    //         $baseFare      = $rate;
    //         $extraFare     = $additionalKMrate * $extraKm;
    //         $hrcharge      = 0;
    //         $extrahrcharge = 0;
    //         $perhrcharge   = 0;
    //         if ($travelled_time_in_hr > $includedHr) {
    //             $extrahrs = max(0, $travelled_time_in_hr - $includedHr);
    //             if ($traveltype_id == 2) {
    //                 if ($traveledKm > 100) {
    //                     $perhrcharge = $package->above100kmhrcharge;
    //                 } elseif ($traveledKm >= 51) {
    //                     $perhrcharge = $package->below100kmhrcharge;
    //                 } else {
    //                     $perhrcharge = $package->below50kmhrcharge;
    //                 }
    //             } else {
    //                 $perhrcharge = $package->per_hours_price;
    //             }
    //             $extrahrcharge = $extrahrs * $perhrcharge;
    //         }
    //         $waitingcharge = 0;
    //         if ($package->local_type == 2) {
    //             $extrahrcharge        = 0;
    //             $isMinutes            = true;
    //             $allowedwaitingmin    = $traveledKm * $setting->waitingminperkm;
    //             $estimatedtime_to_min = $travelled_time_in_hr * 60;
    //             if ($estimatedtime_to_min > $allowedwaitingmin) {
    //                 $waitingcharge = max(0, $estimatedtime_to_min - $allowedwaitingmin) * $setting->waitingchargepermin;
    //             }
    //         }

    //         $totalCost = $baseFare + $extraFare + $hrcharge + $waitingcharge;

    //         $result = ['package_id' => $package->package_id,
    //             'amount'                => $totalCost,
    //             'additional_kmrate'     => $additionalKMrate,
    //             'additionalperhrrate'   => $perhrcharge,
    //             'totalKm'               => $traveledKm,
    //             'baseKm'                => $includedKm,
    //             'extraKm'               => $extraKm,
    //             'totalTime'             => $travelled_time_in_hr,
    //             'freehours'             => $includedHr,
    //             'baseFare'              => $baseFare,
    //             'extraKmFare'           => $extraFare,
    //             'extraHourFare'         => $extrahrcharge,
    //             'waitingcharge'         => $waitingcharge,
    //             'is_minutes'            => $isMinutes ? 1 : 0,
    //         ];

    //         // print_r($result);
    //         // exit();
    //         DB::commit();

    //         $end_date = Date('Y-m-d');

    //         $getexistance = DB::table('fare_summaries')->where('booking_id', $bookingId)->first();
    //         if (! $getexistance) {
    //             $FareSummary = DB::table('fare_summaries')->insertGetId([
    //                 'booking_id'         => $bookingId,
    //                 'driver_id'          => $booking->driver_id,
    //                 'customer_name'      => $tripInfo->customer_name,
    //                 'car_type'           => $tripInfo->car_type,
    //                 'pickup_location'    => $tripInfo->from_address,
    //                 'drop_location'      => $tripInfo->to_address,
    //                 'total_km_travelled' => $result['totalKm'],
    //                 'package_km'         => $result['baseKm'],
    //                 'extra_km'   => $result['extraKm'],
    //                 'base_fare'          => $result['baseFare'],
    //                 'per_km_charge'      => $result['additional_kmrate'],
    //                 'distance_fare'      => $result['extraKmFare'],
    //                 'hr_fare'            => $result['extraHourFare'],
    //                 'waiting_charge'     => $result['waitingcharge'],
    //                 'total_price'        => $result['amount'],
    //                 'travel_minutes'     => $travelled_time_in_hr,
    //                 'end_date'           => $end_date,
    //                 'start_time'         => $startTime->format('H:i:s'),
    //                 'end_time'           => $endTime->format('H:i:s'),
    //             ]);
    //         } else {
    //             DB::table('fare_summaries')->where('booking_id', $bookingId)->update([
    //                 'driver_id'          => $booking->driver_id,
    //                 'customer_name'      => $tripInfo->customer_name,
    //                 'car_type'           => $tripInfo->car_type,
    //                 'pickup_location'    => $tripInfo->from_address,
    //                 'drop_location'      => $tripInfo->to_address,
    //                 'total_km_travelled' => $result['totalKm'],
    //                 'package_km'         => $result['baseKm'],
    //                 'extra_km'   => $result['extraKm'],
    //                 'base_fare'          => $result['baseFare'],
    //                 'per_km_charge'      => $result['additional_kmrate'],
    //                 'distance_fare'      => $result['extraKmFare'],
    //                 'hr_fare'            => $result['extraHourFare'],
    //                 'waiting_charge'     => $result['waitingcharge'],
    //                 'total_price'        => $result['amount'],
    //                 'travel_minutes'     => $travelled_time_in_hr,
    //                 'end_date'           => $end_date,
    //                 'start_time'         => $startTime->format('H:i:s'),
    //                 'end_time'           => $endTime->format('H:i:s'),
    //             ]);
    //         }

    //         $nightfair = $this->nightFareSummary($bookingId);
    //         $this->driverTripDetails($driver->id, $end_date);
    //         $customerPhone = DB::table('booking_management')->where('id', $bookingId)->value('phone');

    //         if ($customerPhone) {
    //             $this->sendFareSummaryToCustomer($customerPhone, [
    //                 'customer_name'      => $tripInfo->customer_name,
    //                 'car_type'           => $tripInfo->car_type,
    //                 'pickup_location'    => $tripInfo->from_address,
    //                 'drop_location'      => $tripInfo->to_address,
    //                 'total_km_travelled' => $result['totalKm'],
    //                 'baseFare'           => $result['baseFare'],
    //                 'distance_fare'      => $result['extraKmFare'],
    //                 'waiting_charge'     => $result['waitingcharge'],
    //                 'total_price'        => $result['amount'],
    //             ]);
    //         }
    //         return response()->json([
    //             'success'      => true,
    //             'message'      => 'Ride ended and fare summary saved successfully.',
    //             'fare_summary' => [
    //                 'customer_name'      => $tripInfo->customer_name,
    //                 'car_type'           => $tripInfo->car_type,
    //                 'pickup_location'    => $tripInfo->from_address,
    //                 'drop_location'      => $tripInfo->to_address,
    //                 'total_km_travelled' => $result['totalKm'],
    //                 'package_km'         => $result['baseKm'],
    //                 'extra_km'   => $result['extraKm'],
    //                 'base_fare'          => $result['baseFare'],
    //                 'per_km_charge'      => $result['additional_kmrate'],
    //                 'distance_fare'      => $result['extraKmFare'],
    //                 'hr_fare'            => $result['extraHourFare'],
    //                 'waiting_charge'     => $result['waitingcharge'],
    //                 'total_price'        => $result['amount'],
    //                 'nightfair_amount'   => $nightfair,
    //                 'travel_minutes'     => $travelled_time_in_hr,
    //                 // 'start_time'         => $startTime->format('h:i A'),
    //                 // 'end_time'           => $endTime->format('h:i A'),
    //             ],
    //         ]);

    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         Log::error('Store Error: ' . $e->getMessage(), [
    //             'file'  => $e->getFile(),
    //             'line'  => $e->getLine(),
    //             'trace' => $e->getTraceAsString(),
    //         ]);
    //         return response()->json([
    //             'message' => 'bFailed to end ride and save fare.',
    //             'error'   => $e->getMessage(),
    //         ], 500);
    //     }
    // }

    public function EndRide(Request $request)
    {
        $timeNow = date('Y-m-d H:i:s');

        DB::beginTransaction();

        try {

            $driver = auth('api_driver')->user();

            $bookingId     = $request->booking_id;
            $customerPhone = DB::table('booking_management')->where('id', $bookingId)->where('deleted_status', 0)->value('phone');
            $customerName  = DB::table('booking_management')->where('id', $bookingId)->where('deleted_status', 0)->value('customer_name');

            if ($bookingId > 0) {

                $booking = DB::table('booking_management')
                    ->select('id', 'cartype_id', 'ac_type', 'traveltype_id', 'driver_id', 'package_id')
                    ->where('id', $bookingId)
                    ->where('deleted_status', 0)
                    ->where('driver_id', $driver->id)
                    ->first();

                if (! $booking) {
                    return response()->json(['success' => false, 'message' => 'Booking or trip data not found'], 404);
                }

            } else {
                return response()->json(['success' => false, 'message' => 'Please Enter Valid Booking Id'], 404);
            }

            // Step 1: Update OTP end details
            $data = [
                'end_km'        => $request->end_km,
                'end_latitude'  => $request->end_latitude,
                'end_longitude' => $request->end_longitude,
                'end_date'      => $timeNow,
                'updated_at'    => now(),
            ];

            if ($request->hasFile('end_image')) {
                $data['end_image'] = $request->file('end_image')->store('end_images', 'public');
            }
            DB::table('otp_codes')->where('booking_id', $bookingId)->update($data);

            // Step 2: Update trip and driver status
            DB::table('booking_management')->where('id', $bookingId)->where('deleted_status', 0)->update([
                'booking_status' => 4, // End
                'trip_status'    => 0,
                'updated_at'     => now(),
            ]);

            DB::table('driver_management')->where('id', $booking->driver_id)->update([
                'trip_status' => 0, // Off trip
                'updated_at'  => now(),
            ]);
            $booking = DB::table('booking_management')
                ->select('id', 'cartype_id', 'ac_type', 'traveltype_id', 'driver_id', 'package_id')
                ->where('id', $bookingId)
                ->where('deleted_status', 0)
                ->where('driver_id', $driver->id)
                ->first();

            DB::table('driver_management')->where('id', $booking->driver_id)->update(['trip_status' => 0, 'updated_at' => now()]);

            // Step 3: Fetch OTP and calculate distance
            $otp = DB::table('otp_codes')->where('booking_id', $bookingId)->first();
            if (! $otp || ! $booking) {
                DB::rollBack();
                if (! $otp || ! $booking) {
                    Log::error('Store Error: No otp');
                } else {
                    Log::error('Store Error: No booking');
                }

                return response()->json(['success' => false, 'message' => 'Booking or trip data not found'], 404);
            }

            $traveledKm    = $totalKm    = round($otp->end_km - $otp->start_km, 2);
            $cartype_id    = $booking->cartype_id;
            $traveltype_id = $booking->traveltype_id;

            $setting = DB::table('settings')->where('id', 1)->first();

            $tripInfo = DB::table('otp_codes as oc')
                ->join('booking_management as bm', 'oc.booking_id', '=', 'bm.id')
                ->join('driver_management as dm', 'bm.driver_id', '=', 'dm.id')
                ->join('vehicle_management as vm', 'dm.vehicle_id', '=', 'vm.id')
                ->join('cartype_management as cm', 'vm.cartype_id', '=', 'cm.id')
            // ->join('packages as pk', 'pk.id', '=', 'bm.package_id')
                ->select(
                    'bm.customer_name',
                    'bm.phone',
                    'bm.from_address',
                    'bm.to_address',
                    'oc.start_km',
                    'oc.end_km',
                    // 'pk.km as included_km',
                    'cm.car_type',
                    'oc.start_date as start_time',
                    'oc.end_date as end_time'
                )
                ->where('oc.booking_id', $bookingId)
                ->first();
            if (! $tripInfo) {
                DB::rollBack();
                Log::error('Store Error: Failed to end ride and save fare.');
                return response()->json(['success' => false, 'message' => 'aFailed to end ride and save fare.'], 404);

                DB::rollBack();
                Log::error('Store Error: No tripInfo');
                return response()->json(['success' => false, 'message' => 'Trip info not found'], 404);
            }

            $startTime = Carbon::parse($tripInfo->start_time);
            $endTime   = Carbon::parse($tripInfo->end_time);
                                                                          // $travelled_time_in_min = $endTime->diffInMinutes($startTime);
            $durationSeconds       = $endTime->diffInSeconds($startTime); //
            $travelled_time_in_min = number_format($durationSeconds / 60, 2, '.', '');
            $travelled_time_in_hr  = $travelled_time_in_min / 60;
            $isMinutes             = false;

            $package = DB::table('packages as p')
                ->join('outstation_package as op', function ($join) use ($cartype_id, $traveltype_id) {
                    $join->on('p.id', '=', 'op.package_id')
                        ->where('op.cartype_id', $cartype_id)
                        ->where('op.inout_type', $traveltype_id);
                })
                ->where('p.traveltype_id', $traveltype_id)
                ->where('p.km', '<=', $traveledKm)
                ->orderBy('p.km', 'desc')
                ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'p.free_hours as free_hours', 'op.ac_price', 'op.additionalperkm_ac_price', 'op.additionalperkm_nonac_price', 'op.nonac_price', 'op.local_type', 'op.per_hours_ac_price', 'op.per_hours_price', 'op.hillsac_price', 'op.hillsnonac_price')
                ->first();

            if (! $package) {

                $package = DB::table('packages as p')
                    ->join('outstation_package as op', function ($join) use ($cartype_id, $traveltype_id) {
                        $join->on('p.id', '=', 'op.package_id')
                            ->where('op.cartype_id', $cartype_id)
                            ->where('op.inout_type', $traveltype_id);
                    })
                    ->where('p.traveltype_id', $traveltype_id)
                    ->orderBy('p.km', 'asc')
                    ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'p.free_hours as free_hours', 'op.ac_price', 'op.additionalperkm_ac_price', 'op.additionalperkm_nonac_price', 'op.nonac_price', 'op.local_type', 'op.per_hours_ac_price', 'op.per_hours_price', 'op.hillsac_price', 'op.hillsnonac_price')
                    ->first();

            }

            if (! $package) {
                DB::rollBack();
                Log::error('Store Error: No Pack Found');
                return response()->json(['success' => false, 'message' => 'No Pack Found'], 404);
            }
            if ($package->local_type == 2 && $travelled_time_in_min > $setting->localtripmaxmin) {
                $package = DB::table('packages as p')
                    ->join('outstation_package as op', function ($join) use ($cartype_id, $traveltype_id) {
                        $join->on('p.id', '=', 'op.package_id')
                            ->where('op.cartype_id', $cartype_id)
                            ->where('op.inout_type', $traveltype_id);
                    })
                    ->where('p.traveltype_id', $traveltype_id)
                    ->where('p.local_type', '!=', 2)
                    ->orderBy('p.km', 'asc')
                    ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'p.free_hours as free_hours', 'op.ac_price', 'op.additionalperkm_ac_price', 'op.additionalperkm_nonac_price', 'op.nonac_price', 'op.local_type', 'op.per_hours_ac_price', 'op.per_hours_price', 'op.hillsac_price', 'op.hillsnonac_price')
                    ->first();
            }
            DB::table('booking_management')->where('id', $bookingId)->where('deleted_status', 0)->update(['package_id' => $package->package_id]);

            $rate             = $booking->ac_type === 'AC' ? $package->ac_price : $package->nonac_price;
            $additionalKMrate = $booking->ac_type === 'AC' ? $package->additionalperkm_ac_price : $package->additionalperkm_nonac_price;
            $additionalhrrate = $booking->ac_type === 'AC' ? $package->per_hours_ac_price : $package->per_hours_price;
            $includedKm       = $package->included_km;
            $includedHr       = $package->free_hours;
            $extraKm          = max(0, $traveledKm - $includedKm);
            $extrahrs         = 0;

            $baseFare      = $rate;
            $extraFare     = $additionalKMrate * $extraKm;
            $hrcharge      = 0;
            $extrahrcharge = 0;
            $perhrcharge   = 0;
            if ($travelled_time_in_hr > $includedHr) {
                $extrahrs = max(0, $travelled_time_in_hr - $includedHr);
                if ($traveltype_id == 2) {
                    if ($traveledKm > 100) {
                        $perhrcharge = $package->above100kmhrcharge;
                    } elseif ($traveledKm >= 51) {
                        $perhrcharge = $package->below100kmhrcharge;
                    } else {
                        $perhrcharge = $package->below50kmhrcharge;
                    }
                } else {
                    $perhrcharge = $package->per_hours_price;
                }
                $extrahrcharge = $extrahrs * $perhrcharge;
            }
            $waitingcharge = 0;
            if ($package->local_type == 2) {
                $extrahrcharge        = 0;
                $isMinutes            = true;
                $allowedwaitingmin    = $traveledKm * $setting->waitingminperkm;
                $estimatedtime_to_min = $travelled_time_in_hr * 60;
                if ($estimatedtime_to_min > $allowedwaitingmin) {
                    $extramins     = max(0, $estimatedtime_to_min - $allowedwaitingmin);
                    $extrahrs      = $extramins / 60;
                    $waitingcharge = $extramins * $setting->waitingchargepermin;
                }
            }

            $totalCost = $baseFare + $extraFare + $extrahrcharge + $waitingcharge;

            $result = ['package_id' => $package->package_id,
                'amount'                => $totalCost,
                'additional_kmrate'     => $additionalKMrate,
                'additionalperhrrate'   => $perhrcharge,
                'totalKm'               => $traveledKm,
                'baseKm'                => $includedKm,
                'extraKm'               => $extraKm,
                'totalTime'             => $travelled_time_in_hr,
                'freehours'             => $includedHr,
                'baseFare'              => $baseFare,
                'extraKmFare'           => $extraFare,
                'extraHourFare'         => $extrahrcharge,
                'waitingcharge'         => $waitingcharge,
                'is_minutes'            => $isMinutes ? 1 : 0,
            ];

            // print_r($result);
            // exit();
            DB::commit();

            $end_date = Date('Y-m-d');
            // Step 1: Calculate total reward points so far
            // Step 1: Attempt to get existing reward points
            $rewardPointsData = DB::table('fare_summaries')
                ->join('booking_management', 'fare_summaries.booking_id', '=', 'booking_management.id')
                ->where('booking_management.phone', $customerPhone)
                ->where('booking_management.customer_name', $customerName)
                ->where('booking_management.deleted_status', 0)
                ->sum('fare_summaries.reward_points');

// Step 2: Add +2 for the current trip
            $currentRewardPoints = $rewardPointsData + 2;

            $getexistance = DB::table('fare_summaries')->where('booking_id', $bookingId)->first();
            if (! $getexistance) {
                $FareSummary = DB::table('fare_summaries')->insertGetId([
                    'booking_id'         => $bookingId,
                    'driver_id'          => $booking->driver_id,
                    'customer_name'      => $tripInfo->customer_name,
                    'phone'              => $tripInfo->phone,
                    'car_type'           => $tripInfo->car_type,
                    'pickup_location'    => $tripInfo->from_address,
                    'drop_location'      => $tripInfo->to_address,
                    'total_km_travelled' => $result['totalKm'],
                    'package_km'         => $result['baseKm'],
                    'extra_km'           => $result['extraKm'],
                    'base_fare'          => $result['baseFare'],
                    'per_km_charge'      => $result['additional_kmrate'],
                    'extra_km_fare'      => $result['extraKmFare'],
                    'hr_fare'            => $result['extraHourFare'],
                    'waiting_charge'     => $result['waitingcharge'],
                    'total_price'        => $result['amount'],
                    'travel_minutes'     => $travelled_time_in_hr,
                    'end_date'           => $end_date,
                    'start_time'         => $startTime->format('H:i:s'),
                    'end_time'           => $endTime->format('H:i:s'),
                    'reward_points'      => $currentRewardPoints,
                ]);
            } else {
                DB::table('fare_summaries')->where('booking_id', $bookingId)->update([
                    'driver_id'          => $booking->driver_id,
                    'customer_name'      => $tripInfo->customer_name,
                    'phone'              => $tripInfo->phone,
                    'car_type'           => $tripInfo->car_type,
                    'pickup_location'    => $tripInfo->from_address,
                    'drop_location'      => $tripInfo->to_address,
                    'total_km_travelled' => $result['totalKm'],
                    'package_km'         => $result['baseKm'],
                    'extra_km'           => $result['extraKm'],
                    'base_fare'          => $result['baseFare'],
                    'per_km_charge'      => $result['additional_kmrate'],
                    'extra_km_fare'      => $result['extraKmFare'],
                    'hr_fare'            => $result['extraHourFare'],
                    'waiting_charge'     => $result['waitingcharge'],
                    'total_price'        => $result['amount'],
                    'travel_minutes'     => $travelled_time_in_hr,
                    'end_date'           => $end_date,
                    'start_time'         => $startTime->format('H:i:s'),
                    'end_time'           => $endTime->format('H:i:s'),
                    'reward_points'      => $currentRewardPoints,
                ]);
            }

            $nightfair = $this->nightFareSummary($bookingId);
            $this->driverTripDetails($driver->id, $end_date);
            $customerPhone = DB::table('booking_management')->where('id', $bookingId)->where('deleted_status', 0)->value('phone');
            // Fetch existing total reward points

            if ($customerPhone) {
                $this->sendFareSummaryToCustomer($customerPhone, [
                    'customer_name'      => $tripInfo->customer_name,
                    'car_type'           => $tripInfo->car_type,
                    'pickup_location'    => $tripInfo->from_address,
                    'drop_location'      => $tripInfo->to_address,
                    'total_km_travelled' => $result['totalKm'],
                    'baseFare'           => $result['baseFare'],
                    'extra_km'           => $result['extraKm'],
                    'extra_km_fare'      => $result['extraKmFare'],
                    'distance_fare'      => $result['extraKmFare'],
                    'waiting_charge'     => $result['waitingcharge'],
                    'total_price'        => $result['amount'],
                    'reward_points'      => $currentRewardPoints,
                ]);
            }
            return response()->json([
                'success'      => true,
                'message'      => 'Ride ended and fare summary saved successfully.',
                'fare_summary' => [
                    'customer_name'      => $tripInfo->customer_name,
                    'car_type'           => $tripInfo->car_type,
                    'pickup_location'    => $tripInfo->from_address,
                    'drop_location'      => $tripInfo->to_address,
                    'total_km_travelled' => $result['totalKm'],
                    'package_km'         => $result['baseKm'],
                    'extra_km'           => $result['extraKm'],
                    'base_fare'          => $result['baseFare'],
                    'per_km_charge'      => $result['additional_kmrate'],
                    'extra_km_fare'      => $result['extraKmFare'],
                    'distance_fare'      => $result['extraKmFare'],
                    'hr_fare'            => $result['extraHourFare'],
                    'waiting_charge'     => $result['waitingcharge'],
                    'total_price'        => $result['amount'],
                    'nightfair_amount'   => $nightfair,
                    'travel_minutes'     => $travelled_time_in_hr,
                    'reward_points'      => $currentRewardPoints,
                    // 'start_time'         => $startTime->format('h:i A'),
                    // 'end_time'           => $endTime->format('h:i A'),
                ],
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Store Error: ' . $e->getMessage(), [
                'file'  => $e->getFile(),
                'line'  => $e->getLine(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'message' => 'Failed to end ride and save fare.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function getEndrideAmountAndPackage(Request $request, array $data)
    {
        $setting    = DB::table('settings')->where('id', 1)->first();
        $bookingId  = $request->booking_id;
        $bookingId  = $data['booking_id'];
        $actualKm   = 0;
        $actualTime = 0;
        if ($bookingId > 0) {
            $otpRecord = DB::table('otp_codes')
                ->where('booking_id', $bookingId)
                ->select('start_km', 'end_km', 'driver_id', 'booking_id', 'start_date', 'end_date')
                ->first();

            if ($otpRecord && $otpRecord->start_km !== null && $otpRecord->end_km !== null) {
                $actualKm = floatval($otpRecord->end_km) - floatval($otpRecord->start_km);

            }
            if ($otpRecord && $otpRecord->start_date !== null && $otpRecord->end_date !== null) {
                $start      = \Carbon\Carbon::parse($otpRecord->start_date);
                $end        = \Carbon\Carbon::parse($otpRecord->end_date);
                $actualTime = $end->diffInMinutes($start) / 60;

            }
            $package = DB::table('packages as p')
                ->join('outstation_package as op', function ($join) use ($data) {
                    $join->on('p.id', '=', 'op.package_id')
                        ->where('op.cartype_id', $data['cartype_id'])
                        ->where('op.inout_type', $data['traveltype_id']);
                })
                ->where('p.traveltype_id', $data['traveltype_id'])
                ->where('p.km', '<=', $actualKm)
                ->orderBy('p.km', 'desc')
                ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'p.free_hours as free_hours', 'op.ac_price', 'op.additionalperkm_ac_price', 'op.additionalperkm_nonac_price', 'op.nonac_price', 'op.local_type')
                ->first();

            if (! $package) {

                $package = DB::table('packages as p')
                    ->join('outstation_package as op', function ($join) use ($data) {
                        $join->on('p.id', '=', 'op.package_id')
                            ->where('op.cartype_id', $data['cartype_id'])
                            ->where('op.inout_type', $data['traveltype_id']);
                    })
                    ->where('p.traveltype_id', $data['traveltype_id'])
                    ->orderBy('p.km', 'asc')
                    ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'p.free_hours as free_hours', 'op.ac_price', 'op.additionalperkm_ac_price', 'op.additionalperkm_nonac_price', 'op.nonac_price', 'op.local_type')
                    ->first();

            }
            if ($package->local_type == 2 && $actualKm < 10) {
                $isMinutes = false;

                // Treat estimated_time as minutes if less than 1 hour
                $estimated_minutes = $actualTime;
                if ($actualTime < 1) {
                    $isMinutes         = true;
                    $estimated_minutes = $actualTime * 60; // Convert to minutes
                }

                $baseFare     = 100; // ₹100 up to 3km
                $additionalKm = max(0, $actualKm - 3);

                $additionalRate = $data['ac_type'] === 'AC' ? 22 : 20;
                $extraFare      = $additionalKm * $additionalRate;

                                                         // estimated_time is considered in minutes directly
                $waitingcharge = $estimated_minutes * 2; // ₹2 per minute

                $totalCost = $baseFare + $extraFare + $waitingcharge;

                return [
                    'package_id'          => $package->package_id,
                    'outstation_packid'   => $package->outstation_packid,
                    'amount'              => $totalCost,
                    'additional_kmrate'   => $additionalRate,
                    'additionalperhrrate' => 0,
                    'extraKm'             => $additionalKm,
                    'waitingcharge'       => $waitingcharge,
                    'baseFare'            => $baseFare,
                    'extraFare'           => $extraFare,
                    'hrcharge'            => 0,

                ];
            }

            $rate             = $data['ac_type'] === 'AC' ? $package->ac_price : $package->nonac_price;
            $additionalKMrate = $data['ac_type'] === 'AC' ? $package->additionalperkm_ac_price : $package->additionalperkm_nonac_price;
            $additionalhrrate = $data['ac_type'] === 'AC' ? $package->additionalperkm_ac_price : $package->additionalperkm_nonac_price;
            $includedKm       = $package->included_km;
            $extraKm          = max(0, $actualKm - $includedKm);

            $baseFare  = $rate;
            $extraFare = $additionalKMrate * $extraKm;
            $hrcharge  = 0;
            if ($actualTime > $package->free_hours) {
                $extrahrs = max(0, $actualTime - $package->free_hours);
                if ($data['traveltype_id'] == 2) {
                    $hrcharge = 0;
                    if ($actualKm > 100) {
                        $hrcharge = $package->above100kmhrcharge;
                    } elseif ($actualKm >= 51) {
                        $hrcharge = $package->below100kmhrcharge;
                    } else {
                        $hrcharge = $package->below50kmhrcharge;
                    }
                } else {
                    $hrcharge = $package->per_hours_price;
                }
            }
            $waitingcharge = 0;
            if ($package->local_type == 2) {
                $allowedwaitingmin    = $includedKm * $setting->waitingminperkm;+
                $estimatedtime_to_min = $actualTime * 60;
                if ($estimatedtime_to_min > $allowedwaitingmin) {
                    $extramins     = max(0, $estimatedtime_to_min - $allowedwaitingmin);
                    $extrahrs      = $extramins / 60;
                    $waitingcharge = $extramins * $setting->waitingchargepermin;
                }
            }

            $totalCost = $baseFare + $extraFare + $hrcharge + $waitingcharge;

            return [
                'package_id'          => $package->package_id,
                'outstation_packid'   => $package->outstation_packid,
                'amount'              => $totalCost,
                'additional_kmrate'   => $additionalKMrate,
                'additionalperhrrate' => $additionalhrrate,
                'extraKm'             => $extraKm,
                'waitingcharge'       => $waitingcharge,
                'baseFare'            => $baseFare,
                'extraFare'           => $extraFare,
                'hrcharge'            => $hrcharge,
                'is_minutes'          => 0,
            ];
        }
    }

    public function tripDetails(Request $request, $tripid)
    {

        try {
            if ($tripid > 0) {

                $driver = auth('api_driver')->user();
                $data   = [];

                // $booking = DB::table('booking_management')
                //     ->where('id', $tripid)
                //     ->where('driver_id', $driver->id)
                //     ->first();

                $booking = DB::table('booking_management as bm')
                    ->Join('driver_management as dm', 'bm.driver_id', '=', 'dm.id')
                    ->Join('vehicle_management as vm', 'dm.vehicle_id', '=', 'vm.id')
                    ->Join('cartype_management as cm', 'vm.cartype_id', '=', 'cm.id')
                    ->leftJoin('packages as pk', 'pk.id', '=', 'bm.package_id')
                    ->leftJoin('otp_codes as oc', 'oc.booking_id', '=', 'bm.id')
                    ->select(
                        'bm.booking_status',
                        'bm.ac_type',
                        'bm.phone',
                        'bm.customer_name',
                        'bm.from_address',
                        'bm.to_address',
                        'bm.estimated_km',
                        'bm.estimated_time',
                        'bm.estimated_amount',
                        'bm.start_latitude',
                        'bm.start_longitude',
                        'bm.end_latitude',
                        'bm.end_longitude',
                        'oc.start_km',
                        'oc.end_km',
                        'pk.km as included_km',
                        'cm.car_type',
                        'oc.created_at as start_time',
                        'oc.updated_at as end_time',
                        DB::raw('oc.end_km - oc.start_km as total_km')
                    )
                    ->where('bm.id', $tripid)
                    ->where('bm.driver_id', $driver->id)
                    ->first();

                if (! $booking) {
                    return response()->json(['success' => false, 'message' => 'Booking or trip data not found'], 404);
                }
                $data['booking'] = $booking;

                $faresummary = DB::table('fare_summaries')
                    ->where('booking_id', $tripid)
                    ->where('driver_id', $driver->id)
                    ->latest('id')
                    ->first();
                // if (! $faresummary) {
                //     return response()->json(['success' => false, 'message' => 'Booking or trip data not found', 'data' => $data], 404);
                // }
                $data['fare'] = $faresummary;
                return response()->json(['success' => true, 'message' => 'Success', 'data' => $data], 200);

                return response()->json(['success' => false, 'message' => 'Please Enter Valid Booking Id'], 404);
            }

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to update ride start data.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function tripDetailsBydriverTripId(Request $request, $tripid)
    {

        // dd($tripid);
        try {
            if ($tripid > 0) {

                $driver = auth('api_driver')->user();
                $data   = [];

                $driverbooking = DB::table('driver_arrived')
                    ->where('id', $tripid)
                    ->where('driver_id', $driver->id)
                    ->first();
                $bookingId = $driverbooking->booking_id;

                $booking = DB::table('booking_management as bm')
                    ->Join('driver_management as dm', 'bm.driver_id', '=', 'dm.id')
                    ->Join('vehicle_management as vm', 'dm.vehicle_id', '=', 'vm.id')
                    ->Join('cartype_management as cm', 'vm.cartype_id', '=', 'cm.id')
                    ->leftJoin('packages as pk', 'pk.id', '=', 'bm.package_id')
                    ->leftJoin('otp_codes as oc', 'oc.booking_id', '=', 'bm.id')
                    ->select(
                        'bm.booking_status',
                        'bm.customer_name',
                        'bm.phone',
                        'bm.from_address',
                        'bm.to_address',
                        'oc.start_km',
                        'oc.end_km',
                        'pk.km as included_km',
                        'cm.car_type',
                        'oc.created_at as start_time',
                        'oc.updated_at as end_time',
                        DB::raw('oc.end_km - oc.start_km as total_km')
                    )
                    ->where('bm.id', $bookingId)
                    ->where('bm.driver_id', $driver->id)
                    ->first();

                if (! $booking) {
                    return response()->json(['success' => false, 'message' => 'Booking or trip data not found'], 404);
                }
                $data['booking'] = $booking;

                $faresummary = DB::table('fare_summaries')
                    ->where('booking_id', $bookingId)
                    ->where('driver_id', $driver->id)
                    ->latest('id')
                    ->first();
                // if (! $faresummary) {
                //     return response()->json(['success' => false, 'message' => 'Booking or trip data not found', 'data' => $data], 404);
                // }
                $data['fare'] = $faresummary;
                return response()->json(['success' => true, 'message' => 'Success', 'data' => $data], 200);

                return response()->json(['success' => false, 'message' => 'Please Enter Valid Booking Id'], 404);
            }

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to update ride start data.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
    public function Estimated_amount(Request $request)
    {
        DB::beginTransaction();

        try {
            $driver    = auth('api_driver')->user();
            $bookingId = $request->booking_id;

            if (! $bookingId) {
                return response()->json(['success' => false, 'message' => 'Please Enter Valid Booking Id'], 404);
            }

            $booking = DB::table('booking_management')
                ->select('id', 'cartype_id', 'ac_type', 'traveltype_id', 'driver_id', 'package_id', 'estimated_km')
                ->where('id', $bookingId)
                ->where('deleted_status', 0)
                ->where('driver_id', $driver->id)
                ->first();

            if (! $booking) {
                return response()->json(['success' => false, 'message' => 'Booking not found'], 404);
            }

            // Get the best matching package + outstation_package
            $selectedPackage = DB::table('packages as p')
                ->join('outstation_package as op', function ($join) use ($booking) {
                    $join->on('p.id', '=', 'op.package_id')
                        ->where('op.cartype_id', $booking->cartype_id)
                        ->where('op.inout_type', $booking->traveltype_id);
                })
                ->where('p.traveltype_id', $booking->traveltype_id)
                ->where('p.city_id', 0)
                ->where('p.km', '<=', $booking->estimated_km)
                ->orderBy('p.km', 'desc')
                ->select('p.id as package_id', 'op.id as outstation_packid', 'p.km as included_km', 'op.ac_price', 'op.nonac_price')
                ->first();

            if (! $selectedPackage) {
                DB::rollBack();
                return response()->json(['success' => false, 'message' => 'Matching package not found'], 404);
            }

            // Determine rate and calculate estimated amount
            $per_km_rate      = $booking->ac_type === 'AC' ? $selectedPackage->ac_price : $selectedPackage->nonac_price;
            $included_km      = $selectedPackage->included_km;
            $extra_km         = max(0, $booking->estimated_km - $included_km);
            $base_fare        = $per_km_rate * $included_km;
            $extra_fare       = $per_km_rate * $extra_km;
            $estimated_amount = $base_fare + $extra_fare;

            // Update booking_management with package_id and amount
            DB::table('booking_management')
                ->where('id', $bookingId)
                ->where('deleted_status', 0)
                ->update([
                    'estimated_packageid' => $selectedPackage->package_id,
                    'estimated_amount'    => $estimated_amount,
                    'updated_at'          => now(),
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Estimated amount calculated and updated successfully.',
                'data'    => [
                    'package_id'       => $selectedPackage->package_id,
                    'estimated_amount' => number_format($estimated_amount, 2, '.', ''),
                    'base_fare'        => number_format($base_fare, 2, '.', ''),
                    'extra_fare'       => number_format($extra_fare, 2, '.', ''),
                    'per_km_rate'      => $per_km_rate,
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }
    private function sendFareSummaryToCustomer($phone, $summary)
    {
        $message = "Hi {$summary['customer_name']}, your ride has ended. Here is your fare summary:\n"
            . "Car Type: {$summary['car_type']}\n"
            . "From: {$summary['pickup_location']}\n"
            . "To: {$summary['drop_location']}\n"
            . "Total KM: {$summary['total_km_travelled']} km\n"
            . "Total Amount: ₹{$summary['total_price']}\n"
            . "Reward Points: ₹{$summary['reward_points']}\n"
            . "Thank you for riding with us!";

        // Replace with your actual SMS or WhatsApp API integration
        Log::info("Fare summary sent to $phone:\n$message");

        // );

        $x = $this->sendWelcomeSms($phone, 'text', $message);

    }

    public function nightFareSummary($bookingId)
    {

        $fare        = DB::table('fare_summaries')->where('booking_id', $bookingId)->first();
        $nightAmount = 0;
        if (! $fare || ! $fare->start_time || ! $fare->total_price) {
            return; // Safely exit if data is incomplete
        }

        $startTime = Carbon::parse($fare->start_time);
        $hour      = (int) $startTime->format('H');

        // Check if time is between 10 PM and 5 AM
        if ($hour >= 22 || $hour < 5) {
            $nightExtra  = $fare->total_price * 0.25;
            $nightAmount = $fare->total_price + $nightExtra;

            DB::table('fare_summaries')
                ->where('booking_id', $bookingId)
                ->update([
                    'nightfair_amount' => round($nightAmount, 2),
                    'updated_at'       => now(),
                ]);

        }
        return round($nightAmount, 2);
    }
    private function updateDriverTripSummary($driverId, $date)
    {
        $baseQuery = DB::table('fare_summaries')
            ->where('driver_id', $driverId)
            ->whereDate('end_date', $date);

        $totalPrice   = (clone $baseQuery)->sum('total_price');
        $num_of_trips = (clone $baseQuery)->count();

        $commissionPercentage = DB::table('settings')->value('commission_percentage') ?? 0;
        $commissionAmount     = round(($totalPrice * $commissionPercentage) / 100, 2);

        $trip = DB::table('trips')
            ->where('driver_id', $driverId)
            ->whereDate('trip_date', $date)
            ->first();

        if (! $trip) {
            // Insert new trip
            DB::table('trips')->insert([
                'driver_id'             => $driverId,
                'trip_date'             => $date,
                'amount'                => $totalPrice,
                'commission_percentage' => $commissionPercentage,
                'commission_amount'     => $commissionAmount,
                'trips_count'           => $num_of_trips,
                'paid_amount'           => 0,
                'payment_status'        => 0,
                'created_at'            => now(),
                'updated_at'            => now(),
            ]);
        } else {
            $newPaymentStatus = 0;

            if ($trip->paid_amount == $commissionAmount) {
                $newPaymentStatus = 2; // Fully Paid
            } elseif ($trip->paid_amount > 0 && $trip->paid_amount < $commissionAmount) {
                $newPaymentStatus = 1; // Partially Paid
            }

            DB::table('trips')
                ->where('id', $trip->id)
                ->update([
                    'amount'                => $totalPrice,
                    'commission_percentage' => $commissionPercentage,
                    'commission_amount'     => $commissionAmount,
                    'trips_count'           => $num_of_trips,
                    'payment_status'        => $newPaymentStatus,
                    'updated_at'            => now(),
                ]);
        }
    }

}
