<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    public function addTripPayment(Request $request)
    {
        $driver = auth('api_driver')->user();

        $request->validate([
            'req_amount'      => 'required|numeric',
            'payment_receipt' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            // 'paid_amount'             => 'required|numeric',
            // 'payment_status_approvedBy' => 'required',
            'trip_date'       => 'required',
        ]);

        try {
            $trip_date = $request->trip_date;
            $driverId  = $driver->id;
            $amount    = $request->req_amount;

            // ✅ Get latest trip record for the driver
            $trip = DB::table('trips')
                ->where('driver_id', $driverId)
                ->where('trip_date', $trip_date) // or use 'id' if trip_date is nullable
                ->first();

            if (! $trip) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'No trip found for this driver',
                ], 404);
            }

            $oneMinuteAgo = Carbon::now()->subMinute();

            $recentDuplicate = DB::table('trip_payment_details')
                ->where('driver_id', $driverId)
                ->where('req_amount', $amount)
                ->where('created_at', '>=', $oneMinuteAgo)
                ->exists();

            if ($recentDuplicate) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Try Again After 2 minutes.',
                ], 409);
            }

            $tripDate = $trip->trip_date;

            $data = [
                'trip_id'                   => $trip->id,
                'req_amount'                => $amount,
                'paid_amount'               => $request->paid_amount,
                'payment_status_approvedBy' => $request->payment_status_approvedBy,
                'created_at'                => $tripDate,
            ];

            if ($request->hasFile('payment_receipt')) {
                $imagePath               = $request->file('payment_receipt')->store('payment_receipt', 'public');
                $data['payment_receipt'] = $imagePath;
            }

            // Insert payment record
            DB::table('trip_payment_details')->insert($data);

            return response()->json([
                'status'    => 'success',
                'message'   => 'Payment added successfully',
                'trip_date' => $tripDate,
            ]);

        } catch (\Exception $e) {
            Log::error('Add Trip Payment Error: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
}
