<?php
namespace App\Traits;

use Illuminate\Http\JsonResponse;

trait ApiResponseTrait
{
    /**
     * Return a success response.
     *
     * @param  mixed  $data
     * @param  string  $message
     * @param  int  $statusCode
     * @return JsonResponse
     */
    public function success($data = null, $message = 'Success', $statusCode = 200): JsonResponse
    {
        return response()->json([
            'status'  => true,
            'message' => $message,
            'data'    => $data,
        ], $statusCode);
    }

    /**
     * Return an error response.
     *
     * @param  string  $message
     * @param  int  $statusCode
     * @param  array  $errors
     * @return JsonResponse
     */
    public function error($message = 'Something went wrong', $statusCode = 400, $errors = []): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'message' => $message,
            'errors'  => $errors,
        ], $statusCode);
    }

    /**
     * Return a created response (201).
     *
     * @param  mixed  $data
     * @param  string  $message
     * @return JsonResponse
     */
    public function created($data, $message = 'Resource created successfully'): JsonResponse
    {
        return response()->json([
            'status'  => true,
            'message' => $message,
            'data'    => $data,
        ], 201);
    }

    /**
     * Return a no content response (204).
     *
     * @param  string  $message
     * @return JsonResponse
     */
    public function noContent($message = 'No content'): JsonResponse
    {
        return response()->json([
            'status'  => true,
            'message' => $message,
        ], 204);
    }

    /**
     * Return a validation error response (422).
     *
     * @param  array  $errors
     * @param  string  $message
     * @return JsonResponse
     */
    public function validationError($errors, $message = 'Validation failed'): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'message' => $message,
            'errors'  => $errors,
        ], 422);
    }

    /**
     * Return an unauthorized response (401).
     *
     * @param  string  $message
     * @return JsonResponse
     */
    public function unauthorized($message = 'Unauthorized'): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'message' => $message,
        ], 401);
    }

    /**
     * Return a forbidden response (403).
     *
     * @param  string  $message
     * @return JsonResponse
     */
    public function forbidden($message = 'Forbidden'): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'message' => $message,
        ], 403);
    }

    /**
     * Return a not found response (404).
     *
     * @param  string  $message
     * @return JsonResponse
     */
    public function notFound($message = 'Resource not found'): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'message' => $message,
        ], 404);
    }

    /**
     * Return an internal server error response (500).
     *
     * @param  string  $message
     * @return JsonResponse
     */
    public function internalError($message = 'Server error'): JsonResponse
    {
        return response()->json([
            'status'  => false,
            'message' => $message,
        ], 500);
    }
}
